"use client";

import { useMemo, useRef } from "react";
import { motion, useInView } from "framer-motion";
import { Clock, Phone, Mail, MapPin } from "lucide-react";
import { useLocale } from "next-intl";
import data from "@/data/home.json";
import { Divider } from "@heroui/react";

const EASE_OUT_EXPO = [0.22, 1, 0.36, 1] as const;

const leftVariants = {
  hidden: { opacity: 0, x: -40 },
  visible: {
    opacity: 1,
    x: 0,
    transition: { duration: 0.7, ease: EASE_OUT_EXPO },
  },
};

const rightVariants = {
  hidden: { opacity: 0, x: 40 },
  visible: {
    opacity: 1,
    x: 0,
    transition: { duration: 0.7, ease: EASE_OUT_EXPO },
  },
};

const branchVariants = {
  hidden: { opacity: 0, y: 20 },
  visible: (i: number) => ({
    opacity: 1,
    y: 0,
    transition: {
      duration: 0.6,
      delay: 0.2 + i * 0.15,
      ease: EASE_OUT_EXPO,
    },
  }),
};

export const ContactBranches = () => {
  const locale = useLocale();
  const sectionRef = useRef<HTMLElement | null>(null);
  const isInView = useInView(sectionRef, { once: true, margin: "-100px" });

  const workingHours = data?.workingHours;
  const contact = data?.contact;

  const title =
    locale === "ar" ? contact?.title_ar ?? "" : contact?.title_en ?? "";
  const subtitle =
    locale === "ar" ? contact?.subtitle_ar ?? "" : contact?.subtitle_en ?? "";
  const hoursTitle =
    locale === "ar"
      ? workingHours?.title_ar ?? ""
      : workingHours?.title_en ?? "";
  const hoursValue =
    locale === "ar"
      ? workingHours?.hours_ar ?? ""
      : workingHours?.hours_en ?? "";

  const branches = useMemo(() => {
    return (contact?.branches ?? []).map((branch) => ({
      title: locale === "ar" ? branch?.name_ar : branch?.name_en,
      phone: branch?.phone,
      email: branch?.email,
      address: locale === "ar" ? branch?.address_ar : branch?.address_en,
    }));
  }, [contact?.branches, locale]);

  return (
    <section
      ref={sectionRef}
      className="custom-container flex justify-between max-md:flex-col gap-10 md:gap-20 py-10 md:py-20"
    >
      {/* Left Section - Text Content */}
      <motion.div
        variants={leftVariants}
        initial="hidden"
        animate={isInView ? "visible" : "hidden"}
        className="space-y-4 w-full md:w-[48%]"
      >
        {/* Title */}
        <h2 className="h3Text text-secondary">Contact Us</h2>

        <p className="bodyText font-bold text-primaryText max-w-xl leading-relaxed">
          Get in touch with our UAE or Greece branch for inquiries, service
          requests, or appointments.
        </p>

        <Divider className="text-border-clr1 my-5 md:my-8" />

        <p className="bodyText text-primary max-w-2xl">
          We are available to support you with all ship management inquiries,
          fleet-related questions, and coordination requests across our two
          operational branches.
        </p>

        {/* Working Hours Box */}

        <div className="w-full flex items-center justify-between gap-4 bg-[#E8F8FF] rounded-2xl p-6 mt-6 md:mt-8">
          <span className="h5Text text-secondary">{hoursTitle}</span>

          <div className="flex items-center gap-2">
            <Clock className="size-5 text-primary" />
            <span className="h6Text text-primary">{hoursValue}</span>
          </div>
        </div>
      </motion.div>

      {/* Right Section - Branch Cards */}
      <motion.div
        variants={rightVariants}
        initial="hidden"
        animate={isInView ? "visible" : "hidden"}
        className="flex flex-col gap-6 w-full md:w-[48%]"
      >
        {branches?.map((branch, index) => (
          <motion.div
            key={`${branch?.title}-${index}`}
            custom={index}
            variants={branchVariants}
            initial="hidden"
            animate={isInView ? "visible" : "hidden"}
            className="bg-primary rounded-3xl p-6 md:p-8 space-y-4"
          >
            {/* Branch Name */}
            <h3 className="h5Text text-white">{branch?.title}</h3>

            {/* Contact Details */}
            <div className="space-y-3">
              {branch?.phone && (
                <div className="flex items-center gap-3">
                  <Phone className="size-5 text-[#E7F4F9] shrink-0" />
                  <p className="bodyText text-[#E7F4F9]">{branch?.phone}</p>
                </div>
              )}

              {branch?.email && (
                <div className="flex items-center gap-3">
                  <Mail className="size-5 text-[#E7F4F9] shrink-0" />
                  <p className="bodyText text-[#E7F4F9]">{branch?.email}</p>
                </div>
              )}

              {branch?.address && (
                <div className="flex items-start gap-3">
                  <MapPin className="size-5 text-[#E7F4F9] shrink-0 mt-0.5" />
                  <p className="bodyText text-[#E7F4F9] leading-relaxed">
                    {branch?.address}
                  </p>
                </div>
              )}
            </div>
          </motion.div>
        ))}
      </motion.div>
    </section>
  );
};
