"use client";

import { useRef } from "react";
import { motion, useInView } from "framer-motion";
import Image from "next/image";
import { useLocale } from "next-intl";

const EASE_OUT_EXPO = [0.22, 1, 0.36, 1] as const;

const headerVariants = {
  hidden: { opacity: 0, y: 20 },
  visible: {
    opacity: 1,
    y: 0,
    transition: { duration: 0.6, ease: EASE_OUT_EXPO },
  },
};

const cardVariants = {
  hidden: { opacity: 0, y: 30 },
  visible: {
    opacity: 1,
    y: 0,
    transition: { duration: 0.6, ease: EASE_OUT_EXPO },
  },
};

const containerVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.15,
    },
  },
};

type TeamMember = {
  name: string;
  role_en: string;
  role_ar: string;
  description_en: string;
  description_ar: string;
  image: string;
};

const teamMembers: TeamMember[] = [
  {
    name: "Elias Markou",
    role_en: "Managing Director",
    role_ar: "المدير الإداري",
    description_en:
      "Master Mariner with over 20 years of experience in ship operations, fleet management, and regulatory compliance. Oversees the company's strategic direction and operational performance.",
    description_ar:
      "ربان بحري رئيسي يتمتع بخبرة تزيد عن 20 عامًا في عمليات السفن وإدارة الأسطول والامتثال التنظيمي. يشرف على التوجه الاستراتيجي والأداء التشغيلي للشركة.",
    image: "/images/home/ship-management.jpg",
  },
  {
    name: "Eng. Marina Petropoulos",
    role_en: "Technical Manager",
    role_ar: "المدير الفني",
    description_en:
      "Marine engineer specializing in vessel maintenance planning, condition assessments, and dry-dock coordination. Ensures all vessels meet class and IMO requirements.",
    description_ar:
      "مهندس بحري متخصص في تخطيط صيانة السفن وتقييمات الحالة وتنسيق الحوض الجاف. يضمن أن جميع السفن تلبي متطلبات الفئة والمنظمة البحرية الدولية.",
    image: "/images/home/ship-management.jpg",
  },
  {
    name: "Ahmed R. Al Mansoor",
    role_en: "Crewing Manager",
    role_ar: "مدير الطاقم",
    description_en:
      "Experienced in global crew recruitment, training coordination, certification management, and MLC compliance. Responsible for manning all vessels efficiently and safely.",
    description_ar:
      "ذو خبرة في التوظيف العالمي للطاقم وتنسيق التدريب وإدارة الشهادات والامتثال لاتفاقية العمل البحري. مسؤول عن توظيف جميع السفن بكفاءة وأمان.",
    image: "/images/home/ship-management.jpg",
  },
];

const TeamCard = ({ member }: { member: TeamMember }) => {
  const locale = useLocale();
  const isArabic = locale === "ar";

  return (
    <motion.article
      variants={cardVariants}
      className="flex-1 min-w-[280px] bg-[#F9F9FB] rounded-3xl px-4.5 pb-8 pt-4.5 flex flex-col items-center text-center gap-4 group"
    >
      {/* Image */}
      <div className="relative size-[142px] rounded-full overflow-hidden shrink-0">
        <Image
          src={member?.image}
          alt={member?.name}
          fill
          className="object-cover  mix-blend-luminosity group-hover:mix-blend-normal primary-transition"
        />
      </div>

      {/* Name */}
      <h4 className="h5Text text-primary">{member?.name}</h4>

      {/* Role */}
      <p className="captionText text-secondary">
        {isArabic ? member?.role_ar : member?.role_en}
      </p>

      {/* Description */}
      <p className="bodyText text-thirdText leading-relaxed">
        {isArabic ? member?.description_ar : member?.description_en}
      </p>
    </motion.article>
  );
};

export const OurTeam = () => {
  const sectionRef = useRef<HTMLElement | null>(null);
  const isInView = useInView(sectionRef, { once: true, margin: "-100px" });
  const locale = useLocale();

  return (
    <section ref={sectionRef} className="custom-container py-5 md:py-10">
      {/* Header */}
      <motion.div
        variants={headerVariants}
        initial="hidden"
        animate={isInView ? "visible" : "hidden"}
        className="text-center mb-12 space-y-4"
      >
        <h2 className="h3Text text-secondary">
          {locale === "ar" ? "فريقنا" : "Our Team"}
        </h2>
        <p className="h6Text text-primary max-w-3xl mx-auto">
          {locale === "ar"
            ? "محترفون ذوو خبرة يضمنون عمليات سفن آمنة ومتوافقة وفعالة."
            : "Experienced professionals ensuring safe, compliant, and efficient vessel operations."}
        </p>
      </motion.div>

      {/* Team Cards */}
      <motion.div
        variants={containerVariants}
        initial="hidden"
        animate={isInView ? "visible" : "hidden"}
        className="flex flex-wrap gap-8 justify-center"
      >
        {teamMembers.map((member, index) => (
          <TeamCard key={index} member={member} />
        ))}
      </motion.div>
    </section>
  );
};
