"use client";

import { useRef } from "react";
import { motion, useInView } from "framer-motion";
import { TextImageSection } from "@/components";
import data from "@/data/home.json";
import { Divider } from "@heroui/react";
import { Dumbbell } from "lucide-react";
import { useLocale } from "next-intl";

const EASE_OUT_EXPO = [0.16, 1, 0.3, 1] as const;

// Extract card data for cleaner rendering
const getCardData = (locale: string) => [
  {
    title:
      locale === "ar"
        ? data?.about?.vision?.title_ar
        : data?.about?.vision?.title_en,
    text:
      locale === "ar"
        ? data?.about?.vision?.text_ar
        : data?.about?.vision?.text_en,
  },
  {
    title:
      locale === "ar"
        ? data?.about?.mission?.title_ar
        : data?.about?.mission?.title_en,
    text:
      locale === "ar"
        ? data?.about?.mission?.text_ar
        : data?.about?.mission?.text_en,
  },
];

// Animation variants
const containerVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.15,
      delayChildren: 0.1,
    },
  },
};

const cardVariants = {
  hidden: {
    opacity: 0,
  },
  visible: {
    opacity: 1,
    transition: {
      duration: 0.7,
      ease: EASE_OUT_EXPO,
    },
  },
};

const iconVariants = {
  hidden: {
    scale: 0,
    rotate: -180,
    opacity: 0,
  },
  visible: {
    scale: 1,
    rotate: 90,
    opacity: 1,
    transition: {
      duration: 0.6,
      ease: EASE_OUT_EXPO,
      delay: 0.15,
    },
  },
};

const titleVariants = {
  hidden: {
    opacity: 0,
    y: 15,
    filter: "blur(4px)",
  },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    transition: {
      duration: 0.5,
      ease: EASE_OUT_EXPO,
      delay: 0.25,
    },
  },
};

const dividerVariants = {
  hidden: {
    scaleX: 0,
    opacity: 0,
  },
  visible: {
    scaleX: 1,
    opacity: 1,
    transition: {
      duration: 0.6,
      ease: EASE_OUT_EXPO,
      delay: 0.35,
    },
  },
};

const textVariants = {
  hidden: {
    opacity: 0,
    y: 15,
  },
  visible: {
    opacity: 1,
    y: 0,
    transition: {
      duration: 0.5,
      ease: EASE_OUT_EXPO,
      delay: 0.45,
    },
  },
};

const Card = ({
  title,
  text,
  index,
}: {
  title: string;
  text: string;
  index: number;
}) => {
  const cardRef = useRef<HTMLDivElement | null>(null);
  const isInView = useInView(cardRef, { once: true, margin: "-50px" });

  return (
    <motion.div
      ref={cardRef}
      initial="hidden"
      animate={isInView ? "visible" : "hidden"}
      variants={cardVariants}
      custom={index}
      className="w-full sm:w-1/2 flex flex-col gap-3 p-5 rounded-[1.25rem] bg-[#F9F9FB] group hover:shadow-xl hover:scale-[1.02] primary-transition"
    >
      {/* Icon */}
      <motion.div
        variants={iconVariants}
        className="flex justify-center items-center rounded-lg size-10 md:size-12 bg-secondary group-hover:bg-primary transition-colors duration-300"
      >
        <Dumbbell className="size-6 text-white rotate-90" />
      </motion.div>

      {/* Title */}
      <motion.h2 variants={titleVariants} className="h4Text mt-1 text-primary">
        {title}
      </motion.h2>

      {/* Divider with animation */}
      <motion.div variants={dividerVariants} className="origin-left">
        <Divider className="bg-[#D6D6D6]" />
      </motion.div>

      {/* Text */}
      <motion.p variants={textVariants} className="bodyText text-thirdText">
        {text}
      </motion.p>
    </motion.div>
  );
};

export const AboutSection = () => {
  const locale = useLocale();
  const sectionRef = useRef<HTMLElement | null>(null);
  const isInView = useInView(sectionRef, {
    once: true,
    margin: "-100px",
    amount: 0.2,
  });

  const cards = getCardData(locale);

  return (
    <>
      <TextImageSection
        title={locale === "ar" ? data?.about?.title_ar : data?.about?.title_en}
        textList={locale === "ar" ? data?.about?.body_ar : data?.about?.body_en}
        image={data?.about?.image}
        className="pt-10 md:pt-20 xl:pt-28"
      />

      <motion.section
        ref={sectionRef}
        initial="hidden"
        animate={isInView ? "visible" : "hidden"}
        variants={containerVariants}
        className="custom-container flex items-stretch justify-between gap-3 md:gap-6 max-sm:flex-col mt-5 md:mt-11"
      >
        {cards?.map((card, index) => (
          <Card key={index} title={card.title} text={card.text} index={index} />
        ))}
      </motion.section>
    </>
  );
};
