// src/lib/meta.ts or similar file
import { Metadata } from "next";
import { OpenGraph } from "next/dist/lib/metadata/types/opengraph-types";
import { getLocale, getTranslations } from "next-intl/server";
import { Locale } from "next-intl";
import locales from "./locales";

export const siteConfig = {
  title: "Hellenic Glamor Ship Management",
  description: {
    en: "Comprehensive ship management solutions serving the UAE and Greece.",
    ar: "حلول متكاملة لإدارة السفن تخدم الإمارات واليونان.",
  },
  siteName: "Hellenic Glamor Ship Management",
  image: "/images/logo.png",
  url:
    process.env.NEXT_PUBLIC_SITE_URL ||
    "https://hellenicglamor-uae.com",
};

export const metaObject = async (
  titleKey?: string,
  descriptionKey?: string,
  openGraph?: OpenGraph,
  overrideImage?: string,
  overrideUrl?: string
): Promise<Metadata> => {
  const [t, locale] = await Promise.all([
    getTranslations("METADATA"),
    getLocale(),
  ]);

  const currentLocale = locale as Locale;

  // Dynamic title
  const fullTitle = titleKey ? `${titleKey} - ${t("WEBSITE")} ` : t("WEBSITE");

  // Dynamic description with fallback
  const translatedDescription = descriptionKey
    ? descriptionKey
    : siteConfig.description[locale as keyof typeof siteConfig.description];

  // Build canonical URL
  const canonicalUrl = overrideUrl
    ? new URL(`/${locale}${overrideUrl}`, siteConfig.url).toString()
    : new URL(`/${locale}`, siteConfig.url).toString();

  // Generate all hreflang links dynamically
  const hreflangLinks = locales.map((loc) => ({
    href: `/${loc}${overrideUrl || ""}`,
    hrefLang: loc,
  }));

  // Determine manifest path based on locale
  const manifestPath = `/manifest/${currentLocale}.json`;

  return {
    metadataBase: new URL(siteConfig.url),
    alternates: {
      canonical: canonicalUrl,
      languages: {
        ...hreflangLinks.reduce<Record<string, string>>(
          (acc, { href, hrefLang }) => {
            acc[hrefLang] = `${href}`;
            return acc;
          },
          {}
        ),
      },
    },
    title: fullTitle,
    description: translatedDescription,
    robots: {
      index: true,
      follow: true,
      nocache: false,
      googleBot: {
        index: true,
        follow: true,
      },
    },
    icons: {
      icon: "/images/logo.png",
      shortcut: "/images/logo.png",
      apple: "/images/logo.png",
      other: {
        rel: "logo",
        url: "/images/logo.png",
      },
    },
    openGraph: {
      ...openGraph,
      title: fullTitle,
      description: translatedDescription,
      url: canonicalUrl,
      siteName: siteConfig.siteName,
      images: [
        {
          url: overrideImage || siteConfig.image,
          alt: t("WEBSITE") || "website",
        },
      ],
      locale,
      type: "website",
    },
    twitter: {
      card: "summary_large_image",
      title: fullTitle,
      description: translatedDescription,
      images: {
        url: overrideImage || siteConfig.image,
        alt: t("WEBSITE") || "website",
      },
    },
    applicationName: siteConfig.siteName,
    formatDetection: {
      telephone: true,
    },
    appleWebApp: {
      title: fullTitle,
      capable: true,
      statusBarStyle: "default",
    },
    manifest: manifestPath,
    other: {
      "mobile-web-app-capable": "yes",
    },
  };
};
