"use client";

import Image from "next/image";
import { motion, useInView } from "framer-motion";
import { useRef } from "react";
import { cn } from "@/utils";

const EASE_OUT_EXPO = [0.22, 1, 0.36, 1] as const;

const containerVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.1,
      delayChildren: 0.2,
    },
  },
};

const textBlockVariants = {
  hidden: { opacity: 0, x: -30 },
  visible: {
    opacity: 1,
    x: 0,
    transition: {
      duration: 0.7,
      ease: EASE_OUT_EXPO,
    },
  },
};

const textBlockReverseVariants = {
  hidden: { opacity: 0, x: 30 },
  visible: {
    opacity: 1,
    x: 0,
    transition: {
      duration: 0.8,
      ease: EASE_OUT_EXPO,
    },
  },
};

const dividerVariants = {
  hidden: { scaleX: 0 },
  visible: {
    scaleX: 1,
    transition: {
      duration: 0.8,
      ease: EASE_OUT_EXPO,
    },
  },
};

const imageVariants = {
  hidden: { opacity: 0, scale: 0.8 },
  visible: {
    opacity: 1,
    scale: 1,
    transition: {
      duration: 0.8,
      ease: EASE_OUT_EXPO,
    },
  },
};

const textItemVariants = {
  hidden: { opacity: 0, y: 15 },
  visible: {
    opacity: 1,
    y: 0,
    transition: {
      duration: 0.5,
      ease: EASE_OUT_EXPO,
    },
  },
};

interface IItem {
  title: string;
  subTitle?: string;
  textList: string[];
  image: string;
  isRowReverse?: boolean;
  titleClassName?: string;
  className?: string;
}

export const TextImageSection = ({
  title,
  subTitle,
  image,
  textList,
  isRowReverse = false,
  titleClassName,
  className,
}: IItem) => {
  const sectionRef = useRef<HTMLElement | null>(null);
  const isInView = useInView(sectionRef, {
    once: true,
    margin: "-100px",
    amount: 0.2,
  });

  return (
    <section ref={sectionRef} className={className}>
      <div
        className={cn(
          "custom-container flex items-center md:justify-between flex-col gap-10 md:gap-20 xl:gap-28",
          isRowReverse ? "md:flex-row-reverse" : "md:flex-row"
        )}
      >
        <motion.div
          variants={containerVariants}
          initial="hidden"
          animate={isInView ? "visible" : "hidden"}
          className="w-full md:w-[55%] max-w-full flex flex-col gap-4 md:gap-8"
        >
          <motion.h2
            variants={
              isRowReverse ? textBlockReverseVariants : textBlockVariants
            }
            className={cn(
              "h3Text text-secondary md:max-w-[30ch]",
              titleClassName
            )}
          >
            {title}
          </motion.h2>

          <motion.div
            variants={dividerVariants}
            className="h-px bg-border-clr1 origin-left"
          />

          {subTitle && (
            <motion.p
              variants={textItemVariants}
              className="h5Text text-primaryText text-start whitespace-pre-line"
            >
              {subTitle}
            </motion.p>
          )}

          <motion.div
            variants={containerVariants}
            className="flex flex-col gap-2 md:gap-4"
          >
            {textList?.map((item, index) => (
              <motion.p
                key={index}
                variants={textItemVariants}
                className="bodyText text-primaryText text-start"
              >
                {item}
              </motion.p>
            ))}
          </motion.div>
        </motion.div>

        <motion.div
          initial="hidden"
          animate={isInView ? "visible" : "hidden"}
          variants={imageVariants}
          className="w-full md:w-1/3 flex max-sm:flex-col justify-between gap-2 md:gap-4"
        >
          <Image
            src={image || "/placeholder.svg"}
            alt={title}
            width={600}
            height={530}
            className="size-full rounded-xl object-cover aspect-[4.5/3.85]"
            loading="lazy"
          />
        </motion.div>
      </div>
    </section>
  );
};
