"use client";

import servicesData from "@/data/services.json";
import { PageHero } from "@/views/shared";
import { TextImageSection } from "@/components/text-image-section";
import { IncludesServices, KeyBenefits } from "@/views/services";
import { CTASection } from "@/views/shared/CtaSection";
import { useLocale } from "next-intl";
import { useMemo } from "react";

const services = servicesData.services;

const getLocalizedValue = (lang: string, en?: string, ar?: string) => {
  if (lang === "ar") return ar ?? en ?? "";
  return en ?? ar ?? "";
};

type Service = (typeof services)[number];

type ServicePageProps = {
  slug: string;
  initialService?: Service;
  lang: string;
};

export const ServicePage = ({ slug, initialService, lang }: ServicePageProps) => {
  const runtimeLocale = useLocale();
  const locale = lang || runtimeLocale;

  const service = useMemo(
    () => initialService ?? services.find((item) => item.slug === slug),
    [initialService, slug]
  );

  if (!service) {
    return null;
  }

  const title = getLocalizedValue(locale, service.title_en, service.title_ar);
  const breadcrumbs = useMemo(
    () => [
      { label: locale === "ar" ? "الرئيسية" : "Home", href: `/${locale}` },
      {
        label: locale === "ar" ? "خدماتنا" : "Services",
        href: `/${locale}/services`,
      },
      { label: title, isCurrent: true },
    ],
    [locale, title]
  );

  const introTitle = getLocalizedValue(
    locale,
    service.intro_title_en,
    service.intro_title_ar
  );
  const introHighlight = getLocalizedValue(
    locale,
    service.intro_highlight_en,
    service.intro_highlight_ar
  );
  const introBody = getLocalizedValue(
    locale,
    service.intro_body_en,
    service.intro_body_ar
  );

  const includesTitle = getLocalizedValue(
    locale,
    service.includes_title_en,
    service.includes_title_ar
  );
  const includes = (service.includes_en || []).map((text_en, index) => ({
    text_en,
    text_ar: service.includes_ar?.[index] ?? text_en,
  }));

  const benefitsTitle = getLocalizedValue(
    locale,
    service.benefits_title_en,
    service.benefits_title_ar
  );
  const benefits = (service.benefits_en || []).map((text_en, index) => ({
    text_en,
    text_ar: service.benefits_ar?.[index] ?? text_en,
  }));

  const whyTitle = getLocalizedValue(
    locale,
    service.why_title_en,
    service.why_title_ar
  );
  const whyBody = getLocalizedValue(
    locale,
    service.why_body_en,
    service.why_body_ar
  );

  const ctaDescription = getLocalizedValue(
    locale,
    service.cta_text_en,
    service.cta_text_ar
  );
  const ctaButton = getLocalizedValue(
    locale,
    service.cta_button_en,
    service.cta_button_ar
  );

  return (
    <>
      <PageHero
        title={title}
        backgroundImage={service.images?.hero ?? "/images/home/slide-1.jpg"}
        breadcrumbs={breadcrumbs}
      />

      <TextImageSection
        className="py-10 md:py-16"
        title={introTitle}
        subTitle={introHighlight}
        textList={[introBody]}
        image={service.images?.intro ?? "/images/home/slide-2.jpg"}
      />

      <IncludesServices title={includesTitle} includes={includes} locale={locale} />

      <KeyBenefits title={benefitsTitle} benefits={benefits} locale={locale} />

      <TextImageSection
        className="py-10 md:py-16"
        title={whyTitle}
        textList={[whyBody]}
        image={service.images?.content ?? "/images/home/slide-3.jpg"}
        isRowReverse
      />

      <CTASection
        title={locale === "ar" ? "ابدأ الآن" : "Get Started"}
        description={ctaDescription}
        buttonLabel={ctaButton}
        buttonHref={`/${locale}/contact`}
        backgroundImage={service.images?.cta ?? "/images/home/slide-1.jpg"}
      />
    </>
  );
};
