"use client";

import { useRef, useMemo } from "react";
import { motion, useInView } from "framer-motion";
import { Divider } from "@heroui/react";
import { useLocale } from "next-intl";
import data from "@/data/home.json";
import { ServiceCard } from "./ServiceCard";

const EASE_OUT_EXPO = [0.22, 1, 0.36, 1] as const;

const headerVariants = {
  hidden: { opacity: 0, y: 20 },
  visible: {
    opacity: 1,
    y: 0,
    transition: { duration: 0.6, ease: EASE_OUT_EXPO },
  },
};

const gridVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.1,
    },
  },
};

const cardVariants = {
  hidden: { opacity: 0, y: 30 },
  visible: {
    opacity: 1,
    y: 0,
    transition: { duration: 0.6, ease: EASE_OUT_EXPO },
  },
};

export const ServicesIntroAndGrid = () => {
  const locale = useLocale();
  const sectionRef = useRef<HTMLElement | null>(null);
  const isInView = useInView(sectionRef, { once: true, margin: "-100px" });

  // Memoize locale-dependent data
  const { services } = useMemo(() => {
    const servicesData = data?.services;
    const isArabic = locale === "ar";

    return {
      title: isArabic ? servicesData?.title_ar : servicesData?.title_en,
      subtitle: isArabic
        ? servicesData?.subtitle_ar
        : servicesData?.subtitle_en,
      services:
        servicesData?.items?.map((item) => ({
          title: isArabic ? item.title_ar : item.title_en,
          description: isArabic ? item.description_ar : item.description_en,
          image: item.image,
          slug: item?.key,
        })) || [],
    };
  }, [locale]);

  return (
    <motion.section
      ref={sectionRef}
      initial="hidden"
      animate={isInView ? "visible" : "hidden"}
      className="custom-container flex flex-col gap-8 py-10 md:py-20"
    >
      {/* Header */}
      <motion.div
        variants={headerVariants}
        className="space-y-6 md:space-y-8 lg:px-8"
      >
        <h2 className="h3Text text-secondary">Ship Management Services</h2>

        <Divider className="bg-secondary" />

        <p className="bodyText text-primary font-bold">
          Integrated ship management solutions supporting safe, efficient, and
          compliant vessel operations.
        </p>

        <p className="bodyText text-primary">
          Hellenic Glamor Ship Management L.L.C provides essential services that
          cover all operational, crewing, supply, maintenance, and dry-dock
          needs across our managed fleet. Each service is designed to support
          vessel performance and regulatory alignment through coordinated
          processes between our UAE and Greece branches.
        </p>
      </motion.div>

      {/* Services Grid */}
      <motion.div
        variants={gridVariants}
        initial="hidden"
        animate={isInView ? "visible" : "hidden"}
        className="grid min-[500px]:grid-cols-2 xl:grid-cols-3 gap-5 md:gap-6 lg:px-8 mt-4 md:mt-6"
      >
        {services?.map((service, index) => (
          <motion.div key={`service-${index}`} variants={cardVariants}>
            <ServiceCard service={service} />
          </motion.div>
        ))}
      </motion.div>
    </motion.section>
  );
};
