"use client";

import { useRef } from "react";
import { motion, useInView } from "framer-motion";
import { ArrowUpRight } from "lucide-react";

// Smoother custom easing - spring-like feel
const SMOOTH_EASE = [0.16, 1, 0.3, 1] as const;

const headerVariants = {
  hidden: { opacity: 0, y: 24, filter: "blur(4px)" },
  visible: {
    opacity: 1,
    y: 0,
    filter: "blur(0px)",
    transition: { duration: 0.7, ease: SMOOTH_EASE },
  },
};

const containerVariants = {
  hidden: {},
  visible: {
    transition: {
      staggerChildren: 0.06, // Faster stagger for smoother flow
      delayChildren: 0.1,
    },
  },
};

const itemVariants = {
  hidden: {
    opacity: 0,
    y: 40,
    scale: 0.98,
  },
  visible: {
    opacity: 1,
    y: 0,
    scale: 1,
    transition: {
      duration: 0.5,
      ease: SMOOTH_EASE,
    },
  },
};

type ServiceInclude = {
  text_en: string;
  text_ar: string;
};

type ServiceIncludesProps = {
  title: string;
  includes: ServiceInclude[];
  locale: string;
};

export const IncludesServices = ({
  title,
  includes,
  locale,
}: ServiceIncludesProps) => {
  const sectionRef = useRef<HTMLElement | null>(null);
  const isInView = useInView(sectionRef, { once: true, margin: "-80px" });

  return (
    <section ref={sectionRef} className="custom-container py-10 md:py-20">
      <div className="grid lg:grid-cols-[minmax(0,0.8fr)_minmax(0,1.2fr)] place-items-center gap-8 lg:gap-12 xl:gap-16">
        {/* Left - Title */}
        <motion.div
          variants={headerVariants}
          initial="hidden"
          animate={isInView ? "visible" : "hidden"}
          style={{ willChange: "transform, opacity, filter" }}
        >
          <h2 className="h3Text text-secondary lg:max-w-[20ch]">{title}</h2>
        </motion.div>

        {/* Right - List */}
        <motion.ul
          variants={containerVariants}
          initial="hidden"
          animate={isInView ? "visible" : "hidden"}
          className="flex flex-col gap-4 w-full"
        >
          {includes.map((item, index) => (
            <motion.li
              key={index}
              variants={itemVariants}
              style={{ willChange: "transform, opacity" }}
              className="flex items-start gap-4 border border-[#E4E4E4] rounded-lg px-4 py-3 group hover:border-secondary hover:shadow-md transition-all duration-300 ease-out"
            >
              {/* Icon Circle */}
              <div className="shrink-0 size-8 rounded-full bg-secondary flex items-center justify-center group-hover:scale-110 transition-transform duration-300 ease-out">
                <ArrowUpRight className="size-4 text-white" />
              </div>

              {/* Text */}
              <p className="bodyText text-primary leading-relaxed pt-0.5">
                {locale === "ar" ? item?.text_ar : item?.text_en}
              </p>
            </motion.li>
          ))}
        </motion.ul>
      </div>
    </section>
  );
};
