"use client";

import { useCallback, useEffect, useState } from "react";
import Image from "next/image";
import useEmblaCarousel from "embla-carousel-react";
import Autoplay from "embla-carousel-autoplay";
import { motion, AnimatePresence } from "framer-motion";
import data from "@/data/home.json";
import { useLocale } from "next-intl";

export const HomeHero = () => {
  const [selectedIndex, setSelectedIndex] = useState(0);
  const locale = useLocale();

  const [emblaRef, emblaApi] = useEmblaCarousel(
    {
      loop: true,
      axis: "y",
    },
    [Autoplay({ delay: 3000, stopOnInteraction: false })]
  );

  const slides = data?.hero?.data ?? [];

  const scrollTo = useCallback(
    (index: number) => {
      if (emblaApi) emblaApi.scrollTo(index);
    },
    [emblaApi]
  );

  const onSelect = useCallback(() => {
    if (!emblaApi) return;
    setSelectedIndex(emblaApi.selectedScrollSnap());
  }, [emblaApi]);

  useEffect(() => {
    if (!emblaApi) return;
    onSelect();
    emblaApi.on("select", onSelect);
    return () => {
      emblaApi.off("select", onSelect);
    };
  }, [emblaApi, onSelect]);

  const currentSlide = slides[selectedIndex] ?? slides[0];

  return (
    <motion.section
      className="p-4 md:p-6"
      initial={{ opacity: 0 }}
      animate={{ opacity: 1 }}
      transition={{ duration: 0.6 }}
    >
      <div className="relative w-full max-w-full h-[80vh] rounded-2xl overflow-hidden bg-primary">
        {/* Embla viewport (background) */}
        <div ref={emblaRef} className="absolute inset-0 h-full overflow-hidden">
          <div className="flex flex-col h-full">
            {slides?.map((slide: any, index: number) => (
              <div
                key={index}
                className="relative flex-[0_0_100%] h-full w-full"
              >
                <Image
                  src={slide?.image}
                  alt={
                    locale === "ar"
                      ? slide?.name_ar ?? ""
                      : slide?.name_en ?? ""
                  }
                  fill
                  quality={100}
                  className="object-cover"
                  priority={index === 0}
                  loading={index === 0 ? "eager" : "lazy"}
                />
              </div>
            ))}
          </div>
        </div>

        {/* Bottom overlay with animated gradient */}
        <motion.div
          className="pointer-events-none absolute inset-x-0 bottom-0 h-full bg-linear-to-t from-black/50 to-black/30"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.8, delay: 0.2 }}
        />

        {/* Foreground content */}
        <div className="relative z-10 flex h-full max-md:flex-col items-center md:items-end justify-end md:justify-between gap-16 md:gap-8 px-6 pb-10 md:px-10 pointer-events-none">
          {/* Text box with staggered animation */}
          <AnimatePresence mode="wait">
            <motion.div
              key={selectedIndex}
              className="flex flex-col gap-4 max-w-xl rounded-2xl bg-[#0A1F2F]/60 backdrop-blur-lg p-6 md:p-9 pointer-events-auto"
              initial={{ opacity: 0, y: 30, scale: 0.95 }}
              animate={{ opacity: 1, y: 0, scale: 1 }}
              exit={{ opacity: 0, y: -30, scale: 0.95 }}
              transition={{
                duration: 0.7,
                ease: [0.25, 0.46, 0.45, 0.94], // easeOutQuad
              }}
            >
              <div className="overflow-hidden">
                <motion.h1
                  className="h3Text text-secondaryText primary-transition"
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  transition={{ duration: 0.7, delay: 0.1, ease: "easeInOut" }}
                >
                  {locale === "ar"
                    ? currentSlide?.name_ar
                    : currentSlide?.name_en}
                </motion.h1>
              </div>
              <div className="overflow-hidden">
                <motion.p
                  className="h6Text text-secondaryText primary-transition"
                  initial={{ opacity: 0 }}
                  animate={{ opacity: 1 }}
                  transition={{ duration: 0.7, delay: 0.2, ease: "easeInOut" }}
                >
                  {locale === "ar"
                    ? currentSlide?.tagline_ar
                    : currentSlide?.tagline_en}
                </motion.p>
              </div>
            </motion.div>
          </AnimatePresence>

          {/* Slide indicators with hover and active animations */}
          <motion.div
            className="flex md:flex-col items-center justify-center gap-3 self-center pointer-events-auto"
            initial={{ opacity: 0, scale: 0.8 }}
            animate={{ opacity: 1, scale: 1 }}
            transition={{ duration: 0.5, delay: 0.4 }}
          >
            {slides.map((_: any, index: number) => (
              <motion.button
                key={index}
                onClick={() => scrollTo(index)}
                className={`w-3 h-3 rounded-full transition-colors ${
                  index === selectedIndex ? "bg-secondary" : "bg-white"
                }`}
                aria-label={`Go to slide ${index + 1}`}
                initial={{ scale: 0 }}
                animate={{
                  scale: index === selectedIndex ? 1.1 : 1,
                }}
                whileHover={{ scale: 1.2 }}
                whileTap={{ scale: 0.9 }}
                transition={{
                  type: "spring",
                  stiffness: 300,
                  damping: 20,
                }}
              />
            ))}
          </motion.div>
        </div>
      </div>
    </motion.section>
  );
};
